﻿/*  Author:     Ben Hall
 *  File:       DefenderAI.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    DefenderAI.cs extends the AICharacter class and overrides the abstract method MakeMove()
 *              using a decision tree that focuses on using defensive actions in the presence of allies,
 *              to boost the defense of itself or its allies.  It may also use abilities like Cover to
 *              protect weakened allies.
 *              If alone, the DefenderAI will use less defensive actions and focus more on the strongest
 *              attack actions available to it.
 *              
 *              The MakeMove() method also uses a random chance to skip the "best" move the AICharacter
 *              can make at each branch, based on the AICharacter's intelligence, with the chance of it
 *              skipping a "best" move decreasing every time the "best" move is skipped.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OnlineGame
{
    public class DefenderAI : AICharacter
    {
        public DefenderAI(String name, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn, int currencyIn)
        {
            this.characterName = name;
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);
            //this.level = DetermineLevel();

            this.currencyCarried = currencyIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;
            this.characterCovered = null;

            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

            this.decisionChance = DetermineDecisionChance();
        }

        /*  MakeMove() selects an action for the AICharacter to take, using a decision tree with a
         *  random chance of skipping the "best move" at each branch (based on the AICharacter's intelligence
         *  score)
         *  
         *  DefenderAI prioritizes protecting its weaker allies with defense boosting skills and the Cover
         *  skill, choosing direct attack options less often.
         */
        public override String MakeMove()
        {
            String messageReturned = "";

            //variables needed to determine if an AI skips a more optimal move
            //initialize decisionChance to the base value, based on intelligence stat
            this.decisionChance = DetermineDecisionChance();
            this.rndDecision = MainPage.rnd.NextDouble();
            //AI may decide to skip more optimal moves, with lower intelligence being more likely to skip moves
            
            bool useBestMove = false;
            useBestMove = (rndDecision < decisionChance);

            int chosenSkill = GetCompHealSkill(); // will return -1 if no healing skill is available for use
            if ((this.GetHealthPercent() < lowHealth) && (chosenSkill >= 0) && (useBestMove))
            {
                //Heal self if at low health
                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
            }
            else
            {
                //redetermine useBestMove with a new random number and increased chance if it was false at the last decision step
                if (!useBestMove)
                {
                    useBestMove = RedetermineUseBestMove();
                } 
                
                if ((MainPage.LivingCompCount() > 1) && (useBestMove))
                {
                    if (this.IsCovering())          //no useBestMove
                    {
                        //20% chance of using best Attack, 10% chance of Debuff, 70% chance of using best Defense
                        double random = MainPage.rnd.NextDouble();
                        if (random < 0.2)
                        {
                            messageReturned = UseBestAttack();
                            //Program.FightCommand(this, Program.player);
                        }
                        else if (random < 0.3)
                        {
                            messageReturned = UseRandomDebuff();
                        }
                        else
                        {
                            //Use best defensive action
                            messageReturned = UseBestDefense();
                        }
                    }
                    else
                    {
                        if (!useBestMove)
                        {
                            useBestMove = RedetermineUseBestMove();
                        }
                        //cover the target with lowHealth that is not already covered
                        AICharacter coverTarget = null;
                        if (useBestMove)
                        {
                            for (int i = 0; i < MainPage.compTeam.Length; i++)
                            {
                                //only target living AICharacters under 25% health, other than self
                                if ((MainPage.compTeam[i].GetCurrentHealth() > 0) && (MainPage.compTeam[i].GetHealthPercent() <= lowHealth) && (!MainPage.compTeam[i].Equals(this)))
                                {
                                    if (coverTarget == null)
                                    {
                                        coverTarget = MainPage.compTeam[i];
                                    }
                                    else if (coverTarget.GetHealthPercent() > MainPage.compTeam[i].GetHealthPercent())
                                    {
                                        coverTarget = MainPage.compTeam[i];
                                    }
                                }
                            }
                        }
                        //coverTarget will be null when there are no AI allies with health below 25%, or when useBestMove is false

                        if ((coverTarget != null) && (this.CanUseSkill(GetSkillWithName("Cover"))) && (!this.IsCovered()) && (!this.IsCovering()))
                        {
                            messageReturned = MainPage.UseSkill(MainPage.skillList[GetSkillWithName("Cover")], this, coverTarget);
                        }
                        else
                        {
                            //40% chance of best attack, 10% chance of debuff, 50% chance of best defense
                            double random = MainPage.rnd.NextDouble();
                            if (random < 0.4)
                            {
                                messageReturned = UseBestAttack();
                            }
                            else if (random < 0.5)
                            {
                                messageReturned = UseRandomDebuff();
                            }
                            else
                            {
                                messageReturned = UseBestDefense();
                            }
                        }
                    }
                }
                else
                {
                    if (!useBestMove)
                    {
                        useBestMove = RedetermineUseBestMove();
                    }                    
                    //heal poison on self or attack
                    chosenSkill = GetCompRemovePoisonSkill();
                    if ((this.poisoned) && (chosenSkill >= 0) && (this.GetHealthPercent() <= halfHealth) && (useBestMove))
                    {
                        //heal poison on self
                        messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                    }
                    else
                    {
                        //90% use best attack, 10% Debuff
                        if (MainPage.rnd.NextDouble() < 0.9)
                        {
                            messageReturned = UseBestAttack();
                        }
                        else
                        {
                            messageReturned = UseRandomDebuff();
                        }
                    }
                }
            }
            return messageReturned;
        }
    }
}
